#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "mr1.h"
#include "m2_function_table.h"
#ifdef __linux__
#include <ieee754.h>
#endif

#ifndef PREFIX
#define PREFIX "/usr/local"
#endif
#define LINE_LENGTH_MAX 1024

static int is_n_bit(long long a, int n);
static int is_unsigned_n_bit(long long a, int n);
static int read_coefficients(FILE *fp, int coefficients[5]);
static int read_domain(FILE *fp, int *mshift, int *estart, double *domain_max);
static int read_table(FILE *fp, int coefficient_table[M2_NUM_RACO_WORDS][6], int exponents[M2_NUM_RAEX_WORDS]);

M2_FUNCTION_TABLE m2_read_function_table(const char *filename){
	M2_FUNCTION_TABLE mft;
	char *s;
	char b[FILENAME_MAX];
	FILE *fp = fopen(filename, "r");
	if(fp != NULL){
		goto exit;
	}
	s = getenv("M2_FUNCTION_TABLE_DIR");
	if(s != NULL){
		sprintf(b, "%s/%s", s, filename);
		fp = fopen(b, "r");
		if(fp != NULL){
			goto exit;
		}
	}
	sprintf(b, PREFIX"/lib/m2/%s", filename);
	fp = fopen(b, "r");
	if(fp != NULL){
		goto exit;
	}
	fprintf(stderr, "m2_read_function_table: %s not found.\n", filename);
	exit(EXIT_FAILURE);
exit:
	read_domain(fp, &mft.mshift, &mft.estart, &mft.domain_max);
	read_table(fp, mft.coefficient_table, mft.exponents);
	fclose(fp);
	return mft;
}

static int read_domain(FILE *fp, int *mshift, int *estart, double *domain_max){
	int xmin_exp, xmax_exp;
	union ieee754_float xmin, xmax;
	double r2min, r2max;
	char b[LINE_LENGTH_MAX];
	while(TRUE){
		fgets(b, sizeof b, fp);
		if(b[0] != '#'){
			break;
		}
	}
	if(sscanf(b, "%lf %lf %lf", domain_max, &r2min, &r2max) != 3){
		return FALSE;
	}
	if(r2min == 0){ /* power mode */
		*estart = 1;
		*mshift = r2max;
	}else{
#ifdef Windows_NT
		xmin.f=(1LL << (40*2-18))*r2min/(*domain_max * *domain_max);
		xmax.f=(1LL << (40*2-18))*r2max/(*domain_max * *domain_max);
#else
		xmin.f = sq(pow(2, 40) / *domain_max) * r2min * pow(2, -18);
		xmax.f = sq(pow(2, 40) / *domain_max) * r2max * pow(2, -18);
#endif
		xmin_exp = xmin.ieee.exponent;
		xmax_exp = xmax.ieee.exponent;
		*mshift = log(xmax_exp + 1 - xmin_exp) / log(2) + 0.5;
		*estart = xmax_exp - ((1 << *mshift) - 1);
	}
	if(0 <= *mshift && *mshift <= 8 && is_unsigned_n_bit(*estart, 8)){
		return TRUE;
	}else{
		return FALSE;
	}
}

static int read_table(FILE *fp, int coefficient_table[M2_NUM_RACO_WORDS][6], int exponents[M2_NUM_RAEX_WORDS]){
	int i;
	for(i = 0; i < M2_NUM_RAEX_WORDS; i ++){
		if(!read_coefficients(fp, coefficient_table[i])){
			return FALSE;
		}
		if(fscanf(fp, "%d", exponents + i) != 1
			 || !is_unsigned_n_bit(exponents[i], 8)){
			return FALSE;
		}
	}
	for(i = M2_NUM_RAEX_WORDS; i < M2_NUM_RACO_WORDS; i ++){
		if(!read_coefficients(fp, coefficient_table[i])){
			return FALSE;
		}
	}
	return TRUE;
}

static int read_coefficients(FILE *fp, int coefficients[5]){
	int j;
	int ws[6] = {27, 24, 20, 16, 11, 6};
	for(j = 0; j < 5; j ++){
		if(fscanf(fp, "%d", coefficients + j) != 1
			 || !is_n_bit(coefficients[j], ws[j])){
			fprintf(stderr, "m2_read_function_table: %d is not %d bits\n", coefficients[j], ws[j]);
			return FALSE;
		}
	}
	if(fscanf(fp, "%d", coefficients + 5) != 1
		 || !is_unsigned_n_bit(coefficients[5], ws[5])){
		fprintf(stderr, "m2_read_function_table: %d is not %d bits\n", coefficients[5], ws[5]);
		return FALSE;
	}
	return TRUE;
}

static int is_n_bit(long long a, int n){
	return -(1 << n - 1) <= a && a < 1 << n - 1;
}

static int is_unsigned_n_bit(long long a, int n){
	return 0 <= a && a < 1 << n;
}
